/**
 * GmailParser - Extracts booking/invoice data from Gmail threads using LLM processing
 * 
 * WORKFLOW:
 * 1. Extract sender email/name using multiple Gmail selector strategies
 * 2. Extract thread content from email bodies
 * 3. Send structured prompt to LLM for intelligent data extraction
 * 4. Parse LLM JSON response and populate state with booking fields
 * 5. Keep raw thread text as fallback data
 */

import Booking from '../db/Booking.js';
import Client from '../db/Client.js';

import { PortalParser } from './parser.js';

// Global CONFIG variable - loaded once when parser initializes
let CONFIG = null;


class GmailParser extends PortalParser {

  constructor() {
    super();
    this.STATE = null;
    this.name = 'GmailParser';
    // Don't initialize config in constructor - it's async and should be done when needed
  }

  /**
   * Initialize global CONFIG - load config.json and validate
   * Throws error if config file not present or invalid
   */
  async _initializeConfig() {
    if (CONFIG) return; // Already loaded

    try {
      const configResponse = await fetch(chrome.runtime.getURL('invoicer_config.json'));
      if (!configResponse.ok) {
        throw new Error(`Config file not found: ${configResponse.status}`);
      }
      CONFIG = await configResponse.json();
      console.log('Gmail parser config loaded successfully');
    } catch (error) {
      console.error('FATAL: Unable to load invoicer_config.json:', error);
      throw new Error('Gmail parser cannot initialize - config file missing or invalid');
    }
  }

  /**
   * Check if current page is a Gmail page
   * @param {string} url - Optional URL to check (defaults to current page)
   * @returns {boolean} True if Gmail page detected
   */
  async checkPageMatch(url) {
    const testUrl = url || window.location.href;
    return testUrl.includes('mail.google.com');
  }

  /**
   * Initialize state with default values for Gmail parser
   * @param {Object} state - State object to initialize with defaults
   */
  async initialize(state) {
    
    this.STATE = state;
    this.STATE.clear();
  }


  /**
   * Main parsing function - extracts booking data from Gmail thread
   * @param {Object} state - State object to populate with extracted data
   */
  async parse(state) {
    try {

      // Update STATE with values from passed state, but keep the State instance
      if (state) {
        // Merge hierarchical structure while preserving methods
        if (state.Client) Object.assign(this.STATE.Client, state.Client);
        if (state.Booking) Object.assign(this.STATE.Booking, state.Booking);
        if (state.Config) Object.assign(this.STATE.Config, state.Config);
      }

      // NECESSARY?  should we implement here?
      // await this.waitUntilReady();

      // NAME AND EMAIL
      // FIXME FIXME FIXME this is not working
      // Extract email and name using multiple Gmail selector strategies
      const emailData = this._extractEmailAndName();
     
      // Set guaranteed fields first (easy wins for LLM context)
      if (emailData.email) this.STATE.Client.email = emailData.email;
      if (emailData.name) {
        this.STATE.Client.name = emailData.name;
        this.STATE.Booking.clientId = emailData.name; // Duplicate name as clientId
      }
      this.STATE.Booking.source = 'gmail';


      // Extract thread content from email bodies
      const threadContent = await this._extractThreadContent();
      
      if (!threadContent?.trim()) {
        console.warn('No email content could be extracted. Try refreshing the page or opening the email thread.');
        return;
      }
      console.log("EXTRACTED EMAIL BLOB:" + threadContent);


      // SEND TO LLM
      // exclude emailData in LLM prompt
      const llmResult = await this._sendToLLM(emailData, threadContent);

      console.log("$$$$$ LLM result: " + llmResult);


      if (llmResult) {
        console.log('LLM processed successfully');
        this._conservativeUpdate( llmResult );
      } else {
        console.warn('LLM unavailable - basic extraction only');
      }



      // CHECKING

      // ALWAYS ensure basic email/name persist (even if LLM overwrote them with null)
      if (emailData.email && !this.STATE.Client.email) this.STATE.Client.email = emailData.email;
      if (emailData.name && !this.STATE.Client.name) {
        this.STATE.Client.name = emailData.name;
        this.STATE.Booking.clientId = emailData.name;
      }
      

      // SAME DAY
      // Auto-complete endDate to match startDate if endDate is missing
      if (this.STATE.Booking.startDate && ! this.STATE.Booking.endDate) {
        this.STATE.Booking.endDate = this.STATE.Booking.startDate;
      }

      // DURATION
      // Calculate duration procedurally if startTime and endTime are available
      if (this.STATE.Booking.startTime && this.STATE.Booking.endTime) {
        let duration = this._calculateDuration(this.STATE.Booking.startTime, this.STATE.Booking.endTime);
        if (duration) this.STATE.Booking.duration = duration;
      }

      // RATES
      // if there is a flatRate -- totalAmount = flatRate
      // else if there is an hourlyRate
      // totalAmount = hourlyRate * duration
      // 
      if (this.STATE.Booking.flatRate) {
        // user may ++totalAmount later -- this is just a default
        this.STATE.Booking.totalAmount = this.STATE.Booking.flatRate;
      
      } else if (this.STATE.Booking.hourlyRate && ! this.STATE.Booking.totalAmount) {

        // Calculate totalAmount before displaying if hourlyRate and duration are available
        const hourlyRate = parseFloat(this.STATE.Booking.hourlyRate);
        const calculatedDuration = parseFloat(this.STATE.Booking.duration);
        if (!isNaN(hourlyRate) && !isNaN(calculatedDuration) && hourlyRate > 0 && calculatedDuration > 0) {
          const total = hourlyRate * calculatedDuration;
          this.STATE.Booking.totalAmount = total.toFixed(2);
        }
      } 

      

    } catch (error) {
      console.error('Gmail parser error:', error);
      // Set minimal fallback data
      this.STATE.Booking = this.STATE.Booking || {};
      this.STATE.Booking.source = 'gmail';
    }

    // Return the state object (sidebar will handle saving)
    return this.STATE;
  }





  /**
   * Extract sender email and name using more specific, reliable selectors.
   * This focuses on the primary sender of the thread, avoiding emails in the message body.
   * @returns {Object} Object with email and name properties (null if not found)
   */
  _extractEmailAndName() {
    try {
      console.log("--- Starting Email/Name Extraction ---");
      let email = null, name = null;

      // Strategy 1: Most reliable selector for the main sender's details in an open email.
      // It targets the element that explicitly contains the sender's info in the header.
      const mainSenderElement = document.querySelector('.gD[email]');
      if (mainSenderElement) {
        email = mainSenderElement.getAttribute('email');
        name = mainSenderElement.getAttribute('name') || mainSenderElement.textContent?.trim();
        console.log(`Strategy 1 ('.gD[email]') found: name='${name}', email='${email}'`);
        return { email, name };
      }
      
      // Strategy 2: Fallback for different UI variations. Looks for a span with an email attribute that is a direct child of a specific header class.
      const headerSenderElement = document.querySelector('.gD > span[email]');
       if (headerSenderElement) {
        email = headerSenderElement.getAttribute('email');
        name = headerSenderElement.textContent?.trim();
        console.log(`Strategy 2 ('.gD > span[email]') found: name='${name}', email='${email}'`);
        return { email, name };
      }

      // Strategy 3: Broader search for any element with an email attribute, but check that it's not inside a quoted message to avoid legacy emails.
      const allEmailElements = Array.from(document.querySelectorAll('[email]'));
      const nonQuotedSender = allEmailElements.find(el => !el.closest('.gmail_quote'));
      if (nonQuotedSender) {
        email = nonQuotedSender.getAttribute('email');
        name = nonQuotedSender.getAttribute('name') || nonQuotedSender.textContent?.trim();
        console.log(`Strategy 3 (non-quoted '[email]') found: name='${name}', email='${email}'`);
        return { email, name };
      }
      
      console.warn("Could not find sender email/name using primary strategies.");
      return { email: null, name: null };
      
    } catch (error) {
      console.error('Error extracting email/name:', error);
      return { email: null, name: null };
    }
  }





  /**
   * Expand all collapsed messages in the thread before extracting content.
   * @returns {string} Combined thread content or empty string
   */
  async _extractThreadContent() {
    try {
      // --- FIX: EXPAND COLLAPSED EMAILS ---
      // This selector targets the "Show trimmed content" (the three dots) button that hides previous messages.
      const expanderButtons = document.querySelectorAll('.adA');
      if (expanderButtons.length > 0) {
        console.log(`Found ${expanderButtons.length} collapsed message sections. Expanding...`);
        expanderButtons.forEach(button => button.click());
        // Wait a moment for the DOM to update after the clicks
        await new Promise(resolve => setTimeout(resolve, 500));
      }
      // --- END FIX ---

      const selectors = [
        '.a3s.aiL', // Current Gmail structure (2024+)
        '.ii.gt'    // Older Gmail selector as a fallback
      ];

      let emailBodies = null;
      let usedSelector = '';

      for (const selector of selectors) {
        emailBodies = document.querySelectorAll(selector);
        if (emailBodies.length > 0) {
          usedSelector = selector;
          console.log(`Found ${emailBodies.length} email bodies using selector: ${selector}`);
          break;
        }
      }

      if (!emailBodies || emailBodies.length === 0) {
        console.warn('No email bodies found with any Gmail selector.');
        return '';
      }

      const threadText = Array.from(emailBodies)
        .map(body => {
          const text = body.innerText?.trim() || '';
          console.log(`Extracted content from one email body: ${text.substring(0, 100)}...`);
          return text;
        })
        .filter(text => text.length > 0)
        .join('\n\n--- EMAIL SEPARATOR ---\n\n');

      console.log(`Successfully extracted full thread content (${threadText.length} chars) using selector: ${usedSelector}`);
      return threadText;

    } catch (error) {
      console.error('Error extracting thread content:', error);
      return '';
    }
  }




  /**
   * Conservatively updates the state with new LLM results.
   * Only fills in fields that are currently empty or whitespace or null.
   */
  _conservativeUpdate(llmResult) {
      const updateIfEmpty = (obj, prop, llmValue) => {
          // Only update if the current value is null, undefined, or an empty string
          if (llmValue && (obj[prop] === null || obj[prop] === undefined || String(obj[prop]).trim() === '')) {
              obj[prop] = llmValue;
          }
      };
      
      // Note: We don't update Client name/email from LLM as procedural extraction is more reliable.
      updateIfEmpty(this.STATE.Client, 'phone', llmResult.Client?.phone);
      updateIfEmpty(this.STATE.Client, 'company', llmResult.Client?.company);
      updateIfEmpty(this.STATE.Client, 'notes', llmResult.Client?.notes);

      // Booking fields
      if (llmResult.Booking) {
        Object.keys(this.STATE.Booking).forEach(key => {
            updateIfEmpty(this.STATE.Booking, key, llmResult.Booking[key]);
        });
      }
  }



  
  /**
   * Send thread content to LLM for intelligent booking data extraction
   * @param {Object} emailData - Object containing email and name
   * @param {string} threadContent - Raw email thread text
   * @returns {Object|null} Parsed booking data or null if failed
   */
  async _sendToLLM(emailData, threadContent) {
    try {
      // Ensure CONFIG is loaded
      await this._initializeConfig();
      
      const llmConfig = CONFIG.llm;
      if (!llmConfig?.baseUrl || !llmConfig?.endpoints?.completions) {
        throw new Error('Invalid LLM configuration');
      }

      // Construct structured prompt for booking data extraction
      const prompt = this._buildLLMPrompt(emailData, threadContent, CONFIG.gmailParser);
      
      // Send request via background script to avoid CORS issues
      const response = await this._sendLLMRequest(llmConfig, prompt);

      console.log('LLM request sent, processing response...');
      console.log('LLM response received:', response);
      console.log('Response ok:', response?.ok);
      console.log('Response data:', response?.data);
      console.log('Content array:', response?.data?.content);

      if (!response?.ok) {
        console.error('LLM request failed:', response?.error || 'Request failed');
        return null;
      }
      
      // Handle Anthropic API response format - content is directly the text
      const contentArray = response.data?.content;
      const firstContent = contentArray?.[0];
      const textContent = firstContent?.text || firstContent;
       
      return textContent ? this._parseLLMResponse(textContent) : null;
      
    } catch (error) {
      // Log detailed error for debugging
      console.error('LLM processing failed with error:', error);
      console.error('Error details:', {
        message: error.message,
        stack: error.stack,
        llmConfigExists: !!CONFIG?.llm,
        baseUrl: CONFIG?.llm?.baseUrl,
        endpoints: CONFIG?.llm?.endpoints
      });
      return null;
    }
  }

  /**
   * Build structured prompt for LLM booking data extraction
   * @param {Object} emailData - Email and name data
   * @param {string} threadContent - Email thread content
   * @param {Object} parserConfig - Gmail parser configuration from config file
   * @returns {string} Formatted prompt for LLM
   */
  _buildLLMPrompt(emailData, threadContent, parserConfig) {
    const knownInfo = [
      emailData.email ? `Email: ${emailData.email}` : '',
      emailData.name ? `Name: ${emailData.name}` : ''
    ].filter(Boolean).join('\n');

    // Get system prompt from config, fallback to basic prompt if not available
    const systemPrompt = parserConfig?.systemPrompt || 'Extract booking information from email and output JSON.';

    return `${systemPrompt}\n\n
Known client info:\n
${knownInfo || 'None provided'}\n\n
Email thread content:\n
${threadContent}
`;
  }

  /**
   * Send LLM request via background script
   * @param {Object} llmConfig - LLM configuration
   * @param {string} prompt - Prompt to send
   * @returns {Object|null} Response from background script
   */
  async _sendLocalLLMRequest(llmConfig, prompt) {
    const llmRequest = {
      url: `${llmConfig.baseUrl}${llmConfig.endpoints.completions}`,
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: {
        model: "liquid/lfm2-1.2b",
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.1,
        max_tokens: 1000
      }
    };

      console.log(`Sending LLM request to ${llmConfig.baseUrl}`);


    return new Promise((resolve) => {
      try {
        chrome.runtime.sendMessage(
          { type: 'leedz_llm_request', request: llmRequest },
          (response) => {
            console.log("LLM response received");
            if (chrome.runtime.lastError) {
              console.error('Chrome runtime error:', chrome.runtime.lastError.message);
              resolve(null);
            } else if (!response) {
              console.log('No response from background script');
              resolve(null);
            } else {
              console.log('Valid response received');
              resolve(response);
            }
          }
        );
      } catch (error) {
        console.error('Exception sending message:', error);
        resolve(null);
      }
    });
  }

  /**
   * Send LLM request using config endpoints
   * @param {Object} llmConfig - LLM configuration
   * @param {string} prompt - Prompt to send
   * @returns {Object|null} Response from LLM API
   */
  async _sendLLMRequest(llmConfig, prompt) {
    const llmRequest = {
      url: `${llmConfig.baseUrl}${llmConfig.endpoints.completions}`,
      method: 'POST',
      headers: {
        'x-api-key': llmConfig['api-key'],
        'anthropic-version': llmConfig['anthropic-version'],
        'content-type': 'application/json',
        'anthropic-dangerous-direct-browser-access': 'true'
      },
      body: {
        model: llmConfig.provider,
        max_tokens: llmConfig.max_tokens,
        messages: [{ role: 'user', content: prompt }]
      }
    };

    console.log(`Sending LLM request to ${llmConfig.baseUrl}`);

    return new Promise((resolve) => {
      try {
        chrome.runtime.sendMessage(
          { type: 'leedz_llm_request', request: llmRequest },
          (response) => {
            console.log("LLM response received");
            if (chrome.runtime.lastError) {
              console.error('Chrome runtime error:', chrome.runtime.lastError.message);
              resolve(null);
            } else if (!response) {
              console.log('No response from background script');
              resolve(null);
            } else {
              console.log('Valid response received');
              resolve(response);
            }
          }
        );
      } catch (error) {
        console.error('Exception sending message:', error);
        resolve(null);
      }
    });
  }

  /**
   * Parse LLM response and extract JSON data
   * @param {string} content - Raw LLM response content
   * @returns {Object|null} Parsed booking data or null
   */
  _parseLLMResponse(content) {
    try {
      console.log('_parseLLMResponse called with content:', content?.substring(0, 200));
      const jsonMatch = content.match(/\{[\s\S]*\}/);
      
      if (jsonMatch) {
        console.log('Matched JSON:', jsonMatch[0].substring(0, 200));
        const parsed = JSON.parse(jsonMatch[0]);
        console.log('Parsed JSON:', parsed);
        
        // Map LLM fields to our structured state
        const mapped = {
          Client: {},
          Booking: {},
          Config: {}
        };
        
        /*
        const clientFields = ['name', 'email', 'phone', 'company', 'notes'];
        const bookingFields = ['description', 'location', 'startDate', 'endDate', 
                             'startTime', 'endTime', 'duration', 'hourlyRate', 
                             'flatRate', 'totalAmount', 'status', 'source'];
        */
        const clientFields = Client.getFieldNames();
        const bookingFields = Booking.getFieldNames();


        // Map fields to appropriate sub-objects
        Object.entries(parsed).forEach(([field, value]) => {
          if (value === 'Not applicable' || value === 'Not specified') {
            value = null;
          }

          if (value !== undefined && value !== null) {
            if (clientFields.includes(field)) {
              mapped.Client[field] = value;
            } else if (bookingFields.includes(field)) {
              mapped.Booking[field] = value;
              // Convert numeric fields
              if (['hourlyRate', 'flatRate', 'totalAmount', 'duration'].includes(field)) {
                mapped.Booking[field] = parseFloat(value) || null;
              }
            }
          }
        });
        
        // Ensure clientId is set from Client.name
        if (mapped.Client.name) {
          mapped.Booking.clientId = mapped.Client.name;
        }

        // Handle date mapping: Prioritize startDate/endDate, then parsed.date
        if (parsed.startDate && parsed.startDate !== 'Not applicable' && parsed.startDate !== 'Not specified') {
          mapped.Booking.startDate = parsed.startDate;
        }
        if (parsed.endDate && parsed.endDate !== 'Not applicable' && parsed.endDate !== 'Not specified') {
          mapped.Booking.endDate = parsed.endDate;
        }
        if (parsed.date && parsed.date !== 'Not applicable' && parsed.date !== 'Not specified') {
          if (!mapped.Booking.startDate) mapped.Booking.startDate = parsed.date;
          if (!mapped.Booking.endDate) mapped.Booking.endDate = parsed.date; // Auto-complete endDate if not explicitly provided
        }
        
        // Smart time correction based on duration: If duration suggests overnight work
        if (mapped.Booking.startTime && mapped.Booking.endTime && mapped.Booking.duration) {
          const duration = parseFloat(mapped.Booking.duration);
          if (!isNaN(duration)) {
            const correctedTimes = this._correctTimesWithDuration(mapped.Booking.startTime, mapped.Booking.endTime, duration);
            if (correctedTimes) {
              mapped.Booking.startTime = correctedTimes.startTime;
              mapped.Booking.endTime = correctedTimes.endTime;
            }
          }
        }
        
        // Handle potential alternate mappings or consolidations
        // if (parsed.serviceDate && !mapped.Booking.startDate) mapped.Booking.startDate = parsed.serviceDate; // Removed as LLM now returns startDate/endDate
        if (parsed.address && !mapped.Booking.location) mapped.Booking.location = parsed.address;
        if (parsed.rate && parsed.rate !== 'Not applicable' && parsed.rate !== 'Not specified' && !mapped.Booking.hourlyRate) {
          mapped.Booking.hourlyRate = parsed.rate; // Use parsed.rate if hourlyRate is not set
        }

        // Special handling for duration, rates, and amounts to ensure they are numbers
        if (mapped.Booking.duration) mapped.Booking.duration = parseFloat(mapped.Booking.duration);
        if (mapped.Booking.hourlyRate) mapped.Booking.hourlyRate = parseFloat(mapped.Booking.hourlyRate);
        if (mapped.Booking.flatRate) mapped.Booking.flatRate = parseFloat(mapped.Booking.flatRate);
        if (mapped.Booking.totalAmount) mapped.Booking.totalAmount = parseFloat(mapped.Booking.totalAmount);

        // Clean up NaN values resulting from parseFloat if original was not a valid number
        if (isNaN(mapped.Booking.duration)) mapped.Booking.duration = null;
        if (isNaN(mapped.Booking.hourlyRate)) mapped.Booking.hourlyRate = null;
        if (isNaN(mapped.Booking.flatRate)) mapped.Booking.flatRate = null;
        if (isNaN(mapped.Booking.totalAmount)) mapped.Booking.totalAmount = null;
        
        console.log('Final mapped object:', mapped);
        return mapped;
      }
      return null;
    } catch (error) {
      console.warn('Failed to parse LLM JSON response');
      return null;
    }
  }

  /**
   * Corrects time interpretation based on duration context
   * If times don't make logical sense with duration, adjust AM/PM interpretation
   * @param {string} startTime - Start time from LLM
   * @param {string} endTime - End time from LLM  
   * @param {number} duration - Duration in hours
   * @returns {Object|null} Corrected times or null if no correction needed
   */
  _correctTimesWithDuration(startTime, endTime, duration) {
    try {
      // Parse times - handle both 24hr format and already formatted times
      const parseTime = (timeStr) => {
        if (/(AM|PM)/i.test(timeStr)) {
          // Already formatted, convert to 24hr for calculation
          const match = timeStr.match(/(\d{1,2}):(\d{2})\s*(AM|PM)/i);
          if (!match) return null;
          
          let hours = parseInt(match[1]);
          const minutes = parseInt(match[2]);
          const period = match[3].toUpperCase();
          
          if (period === 'PM' && hours !== 12) hours += 12;
          if (period === 'AM' && hours === 12) hours = 0;
          
          return { hours, minutes, originalFormat: timeStr };
        } else {
          // 24hr format like "19:00"
          const [hours, minutes] = timeStr.split(':').map(Number);
          return { hours, minutes, originalFormat: timeStr };
        }
      };

      const start = parseTime(startTime);
      const end = parseTime(endTime);
      
      if (!start || !end) return null;

      // Calculate actual duration between the times
      let actualDuration;
      if (end.hours >= start.hours) {
        // Same day
        actualDuration = (end.hours - start.hours) + (end.minutes - start.minutes) / 60;
      } else {
        // Overnight (end time next day)
        actualDuration = (24 - start.hours + end.hours) + (end.minutes - start.minutes) / 60;
      }

      // If actual duration doesn't match expected duration (within 0.5 hour tolerance)
      if (Math.abs(actualDuration - duration) > 0.5) {
        // Try different interpretations
        
        // Case 1: If LLM provided 24hr times but they should be interpreted differently
        if (!/(AM|PM)/i.test(startTime) && !/(AM|PM)/i.test(endTime)) {
          // Try interpreting as: 19:00 = 7:00 AM, 11:00 = 11:00 AM (4 hour duration)
          if (start.hours === 19 && end.hours === 11 && Math.abs(duration - 4) < 0.5) {
            return {
              startTime: '7:00 AM',
              endTime: '11:00 AM'
            };
          }
          
          // Try other common misinterpretations
          // 19:00 = 7:00 PM, but if duration is 4 hours, end should be 11:00 PM
          if (start.hours === 19 && duration === 4) {
            return {
              startTime: '7:00 PM', 
              endTime: '11:00 PM'
            };
          }
        }
      }

      return null; // No correction needed
    } catch (error) {
      console.warn('Error correcting times:', error);
      return null;
    }
  }

  /**
   * Calculate duration as the difference between startTime and endTime
   * @param {string} startTime - Start time (12-hour or 24-hour format)
   * @param {string} endTime - End time (12-hour or 24-hour format)
   * @returns {string|null} Duration in hours as string, or null if invalid
   */
  _calculateDuration(startTime, endTime) {
    // DURATION
    // Calculate duration before displaying if startTime and endTime are available
    let duration;

    if (startTime && endTime) {
      const [startHours, startMinutes] = startTime.split(':').map(Number);
      const [endHours, endMinutes] = endTime.split(':').map(Number);

      const startTotalMinutes = startHours * 60 + (startMinutes || 0);
      const endTotalMinutes = endHours * 60 + (endMinutes || 0);

      if (endTotalMinutes < startTotalMinutes) {
        duration = (24 * 60 - startTotalMinutes) + endTotalMinutes;
      } else {
        duration = endTotalMinutes - startTotalMinutes;
      }

      const durationHours = (duration / 60).toFixed(1);
      const durationNum = parseFloat(durationHours);
      return durationNum.toString();
    }

    return null;
  }
}


export default GmailParser;
